<?php

namespace App\Extensions\MarketingBot\System\Services;

use App\Extensions\MarketingBot\System\Models\ShopifyStore;
use App\Extensions\MarketingBot\System\Models\ShopifyProduct;
use App\Extensions\MarketingBot\System\Models\ShopifyOrder;
use App\Extensions\MarketingBot\System\Models\CampaignConversion;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class ShopifyService
{
    protected ShopifyStore $store;
    protected string $apiVersion = '2024-01';

    public function __construct(ShopifyStore $store)
    {
        $this->store = $store;
    }

    /**
     * Get Shopify API base URL
     */
    protected function getApiUrl(string $endpoint = ''): string
    {
        $shopDomain = $this->store->shop_domain;
        return "https://{$shopDomain}/admin/api/{$this->apiVersion}/{$endpoint}";
    }

    /**
     * Make authenticated API request
     */
    protected function makeRequest(string $method, string $endpoint, array $data = [])
    {
        try {
            $response = Http::withHeaders([
                'X-Shopify-Access-Token' => $this->store->access_token,
                'Content-Type' => 'application/json',
            ])->{$method}($this->getApiUrl($endpoint), $data);

            if ($response->successful()) {
                return $response->json();
            }

            Log::error('Shopify API Error', [
                'endpoint' => $endpoint,
                'status' => $response->status(),
                'body' => $response->body(),
            ]);

            return null;
        } catch (\Exception $e) {
            Log::error('Shopify API Exception', [
                'endpoint' => $endpoint,
                'message' => $e->getMessage(),
            ]);

            return null;
        }
    }

    /**
     * Test connection to Shopify
     */
    public function testConnection(): bool
    {
        $response = $this->makeRequest('get', 'shop.json');
        return $response !== null && isset($response['shop']);
    }

    /**
     * Sync products from Shopify
     */
    public function syncProducts(int $limit = 250): int
    {
        $synced = 0;
        $page = 1;

        do {
            $response = $this->makeRequest('get', "products.json?limit={$limit}&page={$page}");

            if (!$response || !isset($response['products'])) {
                break;
            }

            foreach ($response['products'] as $productData) {
                $this->syncProduct($productData);
                $synced++;
            }

            $page++;
        } while (count($response['products']) === $limit);

        $this->store->markSynced('products');

        return $synced;
    }

    /**
     * Sync single product
     */
    protected function syncProduct(array $productData): ShopifyProduct
    {
        $variant = $productData['variants'][0] ?? null;
        $image = $productData['images'][0] ?? null;

        return ShopifyProduct::updateOrCreate(
            [
                'shopify_store_id' => $this->store->id,
                'shopify_product_id' => $productData['id'],
            ],
            [
                'title' => $productData['title'],
                'description' => $productData['body_html'] ?? null,
                'handle' => $productData['handle'],
                'product_type' => $productData['product_type'] ?? null,
                'vendor' => $productData['vendor'] ?? null,
                'price' => $variant['price'] ?? 0,
                'compare_at_price' => $variant['compare_at_price'] ?? null,
                'inventory_quantity' => $variant['inventory_quantity'] ?? 0,
                'inventory_policy' => $variant['inventory_policy'] ?? null,
                'image_url' => $image['src'] ?? null,
                'images' => $productData['images'] ?? [],
                'variants' => $productData['variants'] ?? [],
                'status' => $productData['status'] ?? 'active',
                'is_active' => $productData['status'] === 'active',
            ]
        );
    }

    /**
     * Sync orders from Shopify
     */
    public function syncOrders(?Carbon $since = null, int $limit = 250): int
    {
        $synced = 0;
        $page = 1;
        $sinceParam = $since ? '&created_at_min=' . $since->toIso8601String() : '';

        do {
            $response = $this->makeRequest('get', "orders.json?limit={$limit}&page={$page}&status=any{$sinceParam}");

            if (!$response || !isset($response['orders'])) {
                break;
            }

            foreach ($response['orders'] as $orderData) {
                $this->syncOrder($orderData);
                $synced++;
            }

            $page++;
        } while (count($response['orders']) === $limit);

        $this->store->markSynced('orders');

        return $synced;
    }

    /**
     * Sync single order
     */
    protected function syncOrder(array $orderData): ShopifyOrder
    {
        // Extract UTM parameters from landing site or note attributes
        $utmParams = $this->extractUtmParameters($orderData);

        $order = ShopifyOrder::updateOrCreate(
            [
                'shopify_store_id' => $this->store->id,
                'shopify_order_id' => $orderData['id'],
            ],
            [
                'order_number' => $orderData['order_number'] ?? null,
                'customer_email' => $orderData['email'] ?? null,
                'customer_name' => $orderData['customer']['first_name'] ?? '' . ' ' . $orderData['customer']['last_name'] ?? '',
                'customer_phone' => $orderData['customer']['phone'] ?? null,
                'total_price' => $orderData['total_price'] ?? 0,
                'subtotal_price' => $orderData['subtotal_price'] ?? 0,
                'total_tax' => $orderData['total_tax'] ?? 0,
                'total_discounts' => $orderData['total_discounts'] ?? 0,
                'currency' => $orderData['currency'] ?? 'USD',
                'financial_status' => $orderData['financial_status'] ?? null,
                'fulfillment_status' => $orderData['fulfillment_status'] ?? null,
                'total_items' => count($orderData['line_items'] ?? []),
                'line_items' => $orderData['line_items'] ?? [],
                'shipping_address' => $orderData['shipping_address'] ?? null,
                'billing_address' => $orderData['billing_address'] ?? null,
                'utm_source' => $utmParams['utm_source'] ?? null,
                'utm_medium' => $utmParams['utm_medium'] ?? null,
                'utm_campaign' => $utmParams['utm_campaign'] ?? null,
                'landing_site' => $orderData['landing_site'] ?? null,
                'referring_site' => $orderData['referring_site'] ?? null,
                'shopify_created_at' => isset($orderData['created_at']) ? Carbon::parse($orderData['created_at']) : null,
                'shopify_updated_at' => isset($orderData['updated_at']) ? Carbon::parse($orderData['updated_at']) : null,
            ]
        );

        // Create campaign conversion if UTM campaign is present
        if (!empty($utmParams['utm_campaign'])) {
            $this->createConversion($order, $utmParams);
        }

        return $order;
    }

    /**
     * Extract UTM parameters from order data
     */
    protected function extractUtmParameters(array $orderData): array
    {
        $utmParams = [];

        // Try to extract from landing_site URL
        if (!empty($orderData['landing_site'])) {
            parse_str(parse_url($orderData['landing_site'], PHP_URL_QUERY) ?? '', $queryParams);
            
            foreach (['utm_source', 'utm_medium', 'utm_campaign', 'utm_term', 'utm_content'] as $param) {
                if (isset($queryParams[$param])) {
                    $utmParams[$param] = $queryParams[$param];
                }
            }
        }

        // Try to extract from note_attributes
        if (!empty($orderData['note_attributes'])) {
            foreach ($orderData['note_attributes'] as $attribute) {
                if (str_starts_with($attribute['name'], 'utm_')) {
                    $utmParams[$attribute['name']] = $attribute['value'];
                }
            }
        }

        return $utmParams;
    }

    /**
     * Create campaign conversion record
     */
    protected function createConversion(ShopifyOrder $order, array $utmParams): void
    {
        // Try to find the campaign by ID or name
        $campaignId = null;
        
        if (is_numeric($utmParams['utm_campaign'])) {
            $campaignId = $utmParams['utm_campaign'];
        } else {
            $campaign = \App\Extensions\MarketingBot\System\Models\MarketingCampaign::where('name', 'like', '%' . $utmParams['utm_campaign'] . '%')->first();
            $campaignId = $campaign?->id;
        }

        if ($campaignId) {
            CampaignConversion::updateOrCreate(
                [
                    'campaign_id' => $campaignId,
                    'shopify_order_id' => $order->id,
                ],
                [
                    'revenue' => $order->total_price,
                    'utm_source' => $utmParams['utm_source'] ?? null,
                    'utm_medium' => $utmParams['utm_medium'] ?? null,
                    'utm_campaign' => $utmParams['utm_campaign'] ?? null,
                    'utm_term' => $utmParams['utm_term'] ?? null,
                    'utm_content' => $utmParams['utm_content'] ?? null,
                    'conversion_type' => 'direct',
                    'converted_at' => $order->shopify_created_at,
                ]
            );
        }
    }

    /**
     * Get shop information
     */
    public function getShopInfo(): ?array
    {
        $response = $this->makeRequest('get', 'shop.json');
        return $response['shop'] ?? null;
    }

    /**
     * Generate tracking URL for campaigns
     */
    public static function generateTrackingUrl(string $baseUrl, int $campaignId, string $source = 'whatsapp'): string
    {
        $params = http_build_query([
            'utm_source' => $source,
            'utm_medium' => 'marketing_bot',
            'utm_campaign' => $campaignId,
        ]);

        $separator = str_contains($baseUrl, '?') ? '&' : '?';
        return $baseUrl . $separator . $params;
    }
}
